"""
Tests for Filter and FilterChain - written FIRST before implementation.

TDD Phase 1: These tests define the expected behavior of Filter and FilterChain.
All tests should FAIL until implementation is complete.
"""
import pytest

# This import will fail until implementation exists
try:
    from aurora.filters import Filter, FilterChain
except ImportError:
    Filter = None
    FilterChain = None
    pytestmark = pytest.mark.skip(reason="Filter/FilterChain not yet implemented")


class TestFilterBasic:
    """Basic filter operation tests"""

    def test_filter_equals(self):
        """Filter with == operator"""
        f = Filter("status", "==", "active")
        assert f.apply({"status": "active"}) == True
        assert f.apply({"status": "inactive"}) == False

    def test_filter_not_equals(self):
        """Filter with != operator"""
        f = Filter("status", "!=", "deleted")
        assert f.apply({"status": "active"}) == True
        assert f.apply({"status": "deleted"}) == False

    def test_filter_less_than(self):
        """Filter with < operator"""
        f = Filter("month", "<", 10)
        assert f.apply({"month": 5}) == True
        assert f.apply({"month": 10}) == False
        assert f.apply({"month": 15}) == False

    def test_filter_less_than_or_equal(self):
        """Filter with <= operator"""
        f = Filter("month", "<=", 22)
        assert f.apply({"month": 22}) == True
        assert f.apply({"month": 21}) == True
        assert f.apply({"month": 23}) == False

    def test_filter_greater_than(self):
        """Filter with > operator"""
        f = Filter("epochs", ">", 10)
        assert f.apply({"epochs": 11}) == True
        assert f.apply({"epochs": 10}) == False

    def test_filter_greater_than_or_equal(self):
        """Filter with >= operator"""
        f = Filter("epochs", ">=", 30)
        assert f.apply({"epochs": 30}) == True
        assert f.apply({"epochs": 31}) == True
        assert f.apply({"epochs": 29}) == False

    def test_filter_in_list(self):
        """Filter with 'in' operator"""
        f = Filter("dataset", "in", ["androzoo", "apigraph"])
        assert f.apply({"dataset": "androzoo"}) == True
        assert f.apply({"dataset": "apigraph"}) == True
        assert f.apply({"dataset": "transcendent"}) == False

    def test_filter_not_in_list(self):
        """Filter with 'not_in' operator"""
        f = Filter("dataset", "not_in", ["test", "debug"])
        assert f.apply({"dataset": "production"}) == True
        assert f.apply({"dataset": "test"}) == False

    def test_filter_matches_regex(self):
        """Filter with 'matches' regex operator"""
        f = Filter("name", "matches", r"hcc_.*_warm")
        assert f.apply({"name": "hcc_mlp_warm"}) == True
        assert f.apply({"name": "hcc_mlp_cold"}) == False

    def test_filter_contains(self):
        """Filter with 'contains' operator"""
        f = Filter("description", "contains", "warm")
        assert f.apply({"description": "HCC warm start"}) == True
        assert f.apply({"description": "HCC cold start"}) == False


class TestFilterMissingField:
    """Tests for handling missing fields"""

    def test_filter_missing_field_returns_false(self):
        """Filter returns False when field is missing"""
        f = Filter("nonexistent", "==", "value")
        assert f.apply({"other_field": "value"}) == False

    def test_filter_missing_field_with_none_check(self):
        """Filter can check for None values"""
        f = Filter("optional", "==", None)
        assert f.apply({"optional": None}) == True
        assert f.apply({"other": "value"}) == False


class TestFilterInvalidOperator:
    """Tests for invalid operators"""

    def test_invalid_operator_raises(self):
        """Invalid operator raises ValueError"""
        with pytest.raises(ValueError, match="Unknown operator"):
            Filter("field", "invalid_op", "value")


class TestFilterFromCallable:
    """Tests for creating filters from functions"""

    def test_from_callable_simple(self):
        """Create filter from simple lambda"""
        f = Filter.from_callable(lambda r: r.get("x", 0) > 5)
        assert f.apply({"x": 10}) == True
        assert f.apply({"x": 3}) == False

    def test_from_callable_complex(self):
        """Create filter from complex function"""
        def custom_filter(record):
            month = record.get("month", 0)
            dataset = record.get("dataset", "")
            cutoffs = {"androzoo": 22, "apigraph": 70}
            return month <= cutoffs.get(dataset, float("inf"))

        f = Filter.from_callable(custom_filter, name="cutoff_filter")
        assert f.apply({"month": 20, "dataset": "androzoo"}) == True
        assert f.apply({"month": 25, "dataset": "androzoo"}) == False
        assert f.apply({"month": 65, "dataset": "apigraph"}) == True


class TestFilterFromDict:
    """Tests for creating filters from config dicts"""

    def test_from_dict_basic(self):
        """Create filter from dict config"""
        f = Filter.from_dict({"field": "month", "op": "<=", "value": 22})
        assert f.apply({"month": 20}) == True
        assert f.apply({"month": 25}) == False

    def test_from_dict_missing_keys(self):
        """Missing required keys raises error"""
        with pytest.raises(KeyError):
            Filter.from_dict({"field": "month"})  # Missing op and value


class TestFilterChainAnd:
    """Tests for FilterChain with AND logic"""

    def test_chain_and_all_pass(self):
        """AND chain passes when all filters pass"""
        chain = FilterChain([
            Filter("month", "<=", 22),
            Filter("dataset", "==", "androzoo")
        ], logic="AND")

        assert chain.apply({"month": 20, "dataset": "androzoo"}) == True

    def test_chain_and_one_fails(self):
        """AND chain fails when any filter fails"""
        chain = FilterChain([
            Filter("month", "<=", 22),
            Filter("dataset", "==", "androzoo")
        ], logic="AND")

        assert chain.apply({"month": 25, "dataset": "androzoo"}) == False
        assert chain.apply({"month": 20, "dataset": "apigraph"}) == False

    def test_chain_and_empty(self):
        """Empty AND chain passes everything"""
        chain = FilterChain([], logic="AND")
        assert chain.apply({"anything": "value"}) == True


class TestFilterChainOr:
    """Tests for FilterChain with OR logic"""

    def test_chain_or_one_passes(self):
        """OR chain passes when any filter passes"""
        chain = FilterChain([
            Filter("dataset", "==", "androzoo"),
            Filter("dataset", "==", "apigraph")
        ], logic="OR")

        assert chain.apply({"dataset": "androzoo"}) == True
        assert chain.apply({"dataset": "apigraph"}) == True

    def test_chain_or_none_pass(self):
        """OR chain fails when no filters pass"""
        chain = FilterChain([
            Filter("dataset", "==", "androzoo"),
            Filter("dataset", "==", "apigraph")
        ], logic="OR")

        assert chain.apply({"dataset": "transcendent"}) == False

    def test_chain_or_empty(self):
        """Empty OR chain fails everything"""
        chain = FilterChain([], logic="OR")
        assert chain.apply({"anything": "value"}) == False


class TestFilterChainImmutability:
    """Tests for FilterChain immutability"""

    def test_add_returns_new_chain(self):
        """add() returns new chain, doesn't mutate"""
        chain1 = FilterChain([Filter("a", "==", 1)])
        chain2 = chain1.add(Filter("b", "==", 2))

        assert chain1 is not chain2
        assert len(chain1) == 1
        assert len(chain2) == 2


class TestFilterChainFromConfig:
    """Tests for creating chains from config"""

    def test_from_config_list(self):
        """Create chain from list of dicts"""
        chain = FilterChain.from_config([
            {"field": "month", "op": "<=", "value": 22},
            {"field": "epochs", "op": ">", "value": 10}
        ])

        assert len(chain) == 2
        assert chain.apply({"month": 20, "epochs": 30}) == True
        assert chain.apply({"month": 25, "epochs": 30}) == False
