"""
Tests for Config System - written FIRST before implementation.

TDD Phase 3: These tests define the expected behavior of ConfigLoader, Config, etc.
All tests should FAIL until implementation is complete.
"""
import pytest
import tempfile
import json
import yaml
from pathlib import Path

# This import will fail until implementation exists
try:
    from aurora.config import (
        ConfigLoader,
        Config,
        ConfigValidator,
        ConfigValidationError,
        ConfiguredLoader
    )
    from aurora.filters import Filter, FilterChain
    from aurora.pipeline import Pipeline
except ImportError:
    ConfigLoader = None
    Config = None
    ConfigValidator = None
    ConfigValidationError = None
    ConfiguredLoader = None
    Filter = None
    FilterChain = None
    Pipeline = None
    pytestmark = pytest.mark.skip(reason="Config system not yet implemented")


class TestConfigLoaderFromFile:
    """Tests for loading config from files"""

    @pytest.fixture
    def yaml_config_file(self):
        """Create temporary YAML config file"""
        config = {
            "schema_mapping": {"Old-Name": "new_name"},
            "filters": [{"field": "x", "op": "==", "value": 1}]
        }
        with tempfile.NamedTemporaryFile(mode='w', suffix='.yaml', delete=False) as f:
            yaml.dump(config, f)
            yield Path(f.name)

    @pytest.fixture
    def json_config_file(self):
        """Create temporary JSON config file"""
        config = {
            "schema_mapping": {"Old-Name": "new_name"},
            "filters": [{"field": "x", "op": "==", "value": 1}]
        }
        with tempfile.NamedTemporaryFile(mode='w', suffix='.json', delete=False) as f:
            json.dump(config, f)
            f.flush()
            yield Path(f.name)

    def test_load_yaml_config(self, yaml_config_file):
        """Load configuration from YAML file"""
        config = ConfigLoader.load(yaml_config_file)

        assert isinstance(config, Config)
        assert config.schema_mapping == {"Old-Name": "new_name"}
        assert len(config.filters) == 1

    def test_load_json_config(self, json_config_file):
        """Load configuration from JSON file"""
        config = ConfigLoader.load(json_config_file)

        assert isinstance(config, Config)
        assert config.schema_mapping == {"Old-Name": "new_name"}

    def test_load_nonexistent_file(self):
        """Raise FileNotFoundError for missing file"""
        with pytest.raises(FileNotFoundError):
            ConfigLoader.load("/nonexistent/config.yaml")

    def test_load_invalid_yaml(self):
        """Raise error for invalid YAML"""
        with tempfile.NamedTemporaryFile(mode='w', suffix='.yaml', delete=False) as f:
            f.write("invalid: yaml: content: [")
            f.flush()
            with pytest.raises(Exception):  # ConfigParseError
                ConfigLoader.load(f.name)

    def test_load_string_path(self, yaml_config_file):
        """Accept string path"""
        config = ConfigLoader.load(str(yaml_config_file))
        assert isinstance(config, Config)


class TestConfigLoaderFromDict:
    """Tests for loading config from dict"""

    def test_from_dict_basic(self):
        """Create config from dictionary"""
        config = ConfigLoader.from_dict({
            "schema_mapping": {"a": "b"},
            "filters": []
        })

        assert isinstance(config, Config)
        assert config.schema_mapping == {"a": "b"}

    def test_from_dict_empty(self):
        """Create config from empty dictionary"""
        config = ConfigLoader.from_dict({})

        assert isinstance(config, Config)
        assert config.schema_mapping == {}
        assert config.filters == []

    def test_from_dict_with_all_fields(self):
        """Create config with all supported fields"""
        config = ConfigLoader.from_dict({
            "schema_mapping": {"a": "b"},
            "filters": [{"field": "x", "op": "==", "value": 1}],
            "transforms": [{"type": "convert_arrays"}],
            "array_fields": ["data"],
            "computed_fields": {"c": "d"},
            "loader_options": {"recursive": True}
        })

        assert config.schema_mapping == {"a": "b"}
        assert len(config.filters) == 1
        assert len(config.transforms) == 1
        assert config.array_fields == ["data"]
        assert config.computed_fields == {"c": "d"}
        assert config.loader_options["recursive"] == True


class TestConfigLoaderFromString:
    """Tests for loading config from string"""

    def test_from_yaml_string(self):
        """Create config from YAML string"""
        yaml_str = """
schema_mapping:
  a: b
filters:
  - field: x
    op: "=="
    value: 1
"""
        config = ConfigLoader.from_string(yaml_str, format="yaml")

        assert config.schema_mapping == {"a": "b"}
        assert len(config.filters) == 1

    def test_from_json_string(self):
        """Create config from JSON string"""
        json_str = '{"schema_mapping": {"a": "b"}}'
        config = ConfigLoader.from_string(json_str, format="json")

        assert config.schema_mapping == {"a": "b"}


class TestConfigValidation:
    """Tests for configuration validation"""

    def test_validate_valid_config(self):
        """Valid config passes validation"""
        config = ConfigLoader.from_dict({
            "schema_mapping": {"a": "b"},
            "filters": [{"field": "x", "op": "==", "value": 1}]
        })

        # Should not raise
        config.validate()

    def test_validate_invalid_operator(self):
        """Invalid filter operator fails validation"""
        with pytest.raises(ConfigValidationError) as exc_info:
            ConfigLoader.from_dict({
                "filters": [{"field": "x", "op": "equals", "value": 1}]
            })

        assert "op" in str(exc_info.value).lower()
        assert "equals" in str(exc_info.value)

    def test_validate_missing_filter_field(self):
        """Missing filter field fails validation"""
        with pytest.raises(ConfigValidationError) as exc_info:
            ConfigLoader.from_dict({
                "filters": [{"op": "==", "value": 1}]  # Missing 'field'
            })

        assert "field" in str(exc_info.value).lower()

    def test_validate_invalid_transform_type(self):
        """Invalid transform type fails validation"""
        with pytest.raises(ConfigValidationError) as exc_info:
            ConfigLoader.from_dict({
                "transforms": [{"type": "invalid_transform"}]
            })

        assert "type" in str(exc_info.value).lower()

    def test_validate_schema_mapping_type(self):
        """schema_mapping must be dict"""
        with pytest.raises(ConfigValidationError):
            ConfigLoader.from_dict({
                "schema_mapping": ["not", "a", "dict"]
            })


class TestConfigValidator:
    """Tests for ConfigValidator class"""

    def test_validate_empty_config(self):
        """Empty config is valid"""
        validator = ConfigValidator()
        errors = validator.validate({})

        assert errors == []

    def test_validate_filter_missing_field(self):
        """Filter without 'field' is invalid"""
        validator = ConfigValidator()
        errors = validator.validate({
            "filters": [{"op": "==", "value": 1}]
        })

        assert len(errors) > 0
        assert any("field" in e.lower() for e in errors)

    def test_validate_filter_missing_op(self):
        """Filter without 'op' is invalid"""
        validator = ConfigValidator()
        errors = validator.validate({
            "filters": [{"field": "x", "value": 1}]
        })

        assert len(errors) > 0
        assert any("op" in e.lower() for e in errors)

    def test_validate_filter_missing_value(self):
        """Filter without 'value' is invalid"""
        validator = ConfigValidator()
        errors = validator.validate({
            "filters": [{"field": "x", "op": "=="}]
        })

        assert len(errors) > 0
        assert any("value" in e.lower() for e in errors)

    def test_validate_filter_invalid_op(self):
        """Filter with invalid operator is invalid"""
        validator = ConfigValidator()
        errors = validator.validate({
            "filters": [{"field": "x", "op": "INVALID", "value": 1}]
        })

        assert len(errors) > 0
        assert any("INVALID" in e for e in errors)

    def test_validate_all_valid_operators(self):
        """All valid operators pass"""
        validator = ConfigValidator()
        valid_ops = ["==", "!=", "<", "<=", ">", ">=", "in", "not_in", "matches", "contains"]

        for op in valid_ops:
            errors = validator.validate({
                "filters": [{"field": "x", "op": op, "value": 1}]
            })
            assert errors == [], f"Operator '{op}' should be valid"

    def test_validate_transform_missing_type(self):
        """Transform without 'type' is invalid"""
        validator = ConfigValidator()
        errors = validator.validate({
            "transforms": [{"fields": ["a"]}]
        })

        assert len(errors) > 0
        assert any("type" in e.lower() for e in errors)


class TestConfigToPipeline:
    """Tests for Config.to_pipeline()"""

    def test_empty_config_to_pipeline(self):
        """Empty config creates empty pipeline"""
        config = ConfigLoader.from_dict({})
        pipeline = config.to_pipeline()

        assert isinstance(pipeline, Pipeline)
        assert len(pipeline) == 0

    def test_schema_mapping_to_pipeline(self):
        """schema_mapping creates rename step"""
        config = ConfigLoader.from_dict({
            "schema_mapping": {"old": "new"}
        })
        pipeline = config.to_pipeline()

        result = pipeline.apply([{"old": 1}])
        assert "new" in result[0]
        assert "old" not in result[0]

    def test_filters_to_pipeline(self):
        """filters creates filter step"""
        config = ConfigLoader.from_dict({
            "filters": [{"field": "x", "op": ">", "value": 5}]
        })
        pipeline = config.to_pipeline()

        result = pipeline.apply([{"x": 3}, {"x": 7}])
        assert len(result) == 1
        assert result[0]["x"] == 7

    def test_array_fields_to_pipeline(self):
        """array_fields creates convert_arrays step"""
        import numpy as np

        config = ConfigLoader.from_dict({
            "array_fields": ["data"]
        })
        pipeline = config.to_pipeline()

        result = pipeline.apply([{"data": [1, 2, 3]}])
        assert isinstance(result[0]["data"], np.ndarray)

    def test_transforms_to_pipeline(self):
        """transforms creates multiple steps"""
        config = ConfigLoader.from_dict({
            "transforms": [
                {"type": "rename_fields", "mapping": {"a": "b"}},
                {"type": "remove_fields", "fields": ["temp"]}
            ]
        })
        pipeline = config.to_pipeline()

        result = pipeline.apply([{"a": 1, "temp": 2}])
        assert "b" in result[0]
        assert "temp" not in result[0]

    def test_computed_fields_to_pipeline(self):
        """computed_fields creates computed field steps"""
        # For now, computed_fields just copies field values
        config = ConfigLoader.from_dict({
            "computed_fields": {"copy_of_x": "x"}
        })
        pipeline = config.to_pipeline()

        result = pipeline.apply([{"x": 5}])
        assert result[0]["copy_of_x"] == 5


class TestConfigToFilterChain:
    """Tests for Config.to_filter_chain()"""

    def test_empty_filters_returns_none(self):
        """No filters returns None"""
        config = ConfigLoader.from_dict({})
        chain = config.to_filter_chain()

        assert chain is None

    def test_filters_to_chain(self):
        """Filters converted to FilterChain"""
        config = ConfigLoader.from_dict({
            "filters": [
                {"field": "x", "op": ">=", "value": 0},
                {"field": "x", "op": "<", "value": 10}
            ]
        })
        chain = config.to_filter_chain()

        assert isinstance(chain, FilterChain)
        assert len(chain) == 2
        assert chain.apply({"x": 5}) == True
        assert chain.apply({"x": -1}) == False
        assert chain.apply({"x": 15}) == False


class TestConfigGet:
    """Tests for Config.get()"""

    def test_get_existing_key(self):
        """Get existing top-level key"""
        config = ConfigLoader.from_dict({
            "schema_mapping": {"a": "b"}
        })

        assert config.get("schema_mapping") == {"a": "b"}

    def test_get_missing_key_default(self):
        """Get missing key returns default"""
        config = ConfigLoader.from_dict({})

        assert config.get("missing") is None
        assert config.get("missing", "default") == "default"

    def test_get_nested_key(self):
        """Get nested key with dot notation"""
        config = ConfigLoader.from_dict({
            "loader_options": {"recursive": True}
        })

        assert config.get("loader_options.recursive") == True
        assert config.get("loader_options.missing", False) == False


class TestConfiguredLoader:
    """Tests for ConfiguredLoader"""

    @pytest.fixture
    def temp_data_dir(self):
        """Create temp directory with test data files"""
        import pickle

        with tempfile.TemporaryDirectory() as tmpdir:
            tmpdir = Path(tmpdir)

            # Create test pickle files
            data1 = [
                {"month": 10, "value": 1, "Predictions": [1, 0, 1]},
                {"month": 25, "value": 2, "Predictions": [0, 1, 0]}
            ]
            with open(tmpdir / "data1.pkl", "wb") as f:
                pickle.dump(data1, f)

            data2 = [
                {"month": 15, "value": 3, "Predictions": [1, 1, 0]}
            ]
            with open(tmpdir / "data2.pkl", "wb") as f:
                pickle.dump(data2, f)

            yield tmpdir

    @pytest.fixture
    def config_file(self, temp_data_dir):
        """Create config file"""
        config = {
            "filters": [{"field": "month", "op": "<=", "value": 22}],
            "array_fields": ["Predictions"]
        }
        config_path = temp_data_dir / "config.yaml"
        with open(config_path, 'w') as f:
            yaml.dump(config, f)
        yield config_path

    def test_from_config_file(self, config_file):
        """Create loader from config file"""
        loader = ConfiguredLoader.from_config(config_file)

        assert isinstance(loader, ConfiguredLoader)

    def test_from_config_object(self):
        """Create loader from Config object"""
        config = ConfigLoader.from_dict({"filters": []})
        loader = ConfiguredLoader.from_config_object(config)

        assert isinstance(loader, ConfiguredLoader)

    def test_load_directory(self, temp_data_dir, config_file):
        """Load data from directory"""
        import numpy as np

        loader = ConfiguredLoader.from_config(config_file)
        results = loader.load(temp_data_dir, pattern="*.pkl")

        # Should have filtered out month > 22
        assert all(r["month"] <= 22 for r in results)
        # Should have converted arrays
        assert all(isinstance(r["Predictions"], np.ndarray) for r in results)

    def test_load_file(self, temp_data_dir, config_file):
        """Load single file"""
        loader = ConfiguredLoader.from_config(config_file)
        results = loader.load_file(temp_data_dir / "data1.pkl")

        # Should have filtered
        assert len(results) == 1
        assert results[0]["month"] == 10

    def test_load_with_custom_pattern(self, temp_data_dir, config_file):
        """Load with custom file pattern"""
        loader = ConfiguredLoader.from_config(config_file)
        results = loader.load(temp_data_dir, pattern="data1.pkl")

        # Should only load data1.pkl
        assert len(results) == 1


class TestConfiguredLoaderWithSchema:
    """Tests for ConfiguredLoader with schema mapping"""

    @pytest.fixture
    def temp_data_with_old_names(self):
        """Create temp data with old field names"""
        import pickle

        with tempfile.TemporaryDirectory() as tmpdir:
            tmpdir = Path(tmpdir)

            data = [
                {"Test-Month": 10, "Old-Value": 1},
                {"Test-Month": 20, "Old-Value": 2}
            ]
            with open(tmpdir / "data.pkl", "wb") as f:
                pickle.dump(data, f)

            yield tmpdir

    def test_load_with_schema_mapping(self, temp_data_with_old_names):
        """Load applies schema mapping"""
        config = ConfigLoader.from_dict({
            "schema_mapping": {
                "Test-Month": "month",
                "Old-Value": "value"
            }
        })
        loader = ConfiguredLoader.from_config_object(config)
        results = loader.load(temp_data_with_old_names, pattern="*.pkl")

        assert all("month" in r for r in results)
        assert all("value" in r for r in results)
        assert all("Test-Month" not in r for r in results)
