"""
Composable filtering system for Aurora framework.

This module provides general-purpose filtering mechanisms that work with
any dictionary-based data. No dataset-specific code is included.

Example:
    >>> f = Filter("month", "<=", 22)
    >>> f.apply({"month": 20})  # True
    >>> f.apply({"month": 25})  # False

    >>> chain = FilterChain([
    ...     Filter("month", "<=", 22),
    ...     Filter("dataset", "==", "androzoo")
    ... ])
    >>> chain.apply({"month": 20, "dataset": "androzoo"})  # True
"""

from __future__ import annotations

import re
from typing import Any, Callable, Dict, List, Literal, Optional, Union


class Filter:
    """
    A single filter condition.

    Supports operators: ==, !=, <, <=, >, >=, in, not_in, matches, contains

    Attributes:
        field: Field name to filter on
        operator: Comparison operator
        value: Value to compare against

    Example:
        >>> f = Filter("month", "<=", 22)
        >>> f.apply({"month": 20})  # True
        >>> f.apply({"month": 25})  # False
    """

    # Supported operators with their implementations
    OPERATORS: Dict[str, Callable[[Any, Any], bool]] = {
        "==": lambda a, b: a == b,
        "!=": lambda a, b: a != b,
        "<": lambda a, b: a < b,
        "<=": lambda a, b: a <= b,
        ">": lambda a, b: a > b,
        ">=": lambda a, b: a >= b,
        "in": lambda a, b: a in b,
        "not_in": lambda a, b: a not in b,
        "matches": lambda a, b: bool(re.match(b, str(a))),
        "contains": lambda a, b: b in a,
    }

    def __init__(
        self,
        field: str,
        operator: str,
        value: Any,
        *,
        _callable: Optional[Callable[[Dict], bool]] = None,
        _name: Optional[str] = None,
    ):
        """
        Create a filter condition.

        Args:
            field: Field name to filter on
            operator: Comparison operator (==, !=, <, <=, >, >=, in, not_in, matches, contains)
            value: Value to compare against
            _callable: Internal - custom callable for from_callable()
            _name: Internal - custom name for from_callable()

        Raises:
            ValueError: If operator is not recognized

        Example:
            >>> f = Filter("month", "<=", 22)
            >>> f = Filter("dataset", "==", "androzoo")
            >>> f = Filter("trainer_mode", "in", ["CE", "HCCMLP"])
        """
        if _callable is None and operator not in self.OPERATORS:
            valid_ops = ", ".join(sorted(self.OPERATORS.keys()))
            raise ValueError(f"Unknown operator: '{operator}'. Valid operators: {valid_ops}")

        self.field = field
        self.operator = operator
        self.value = value
        self._callable = _callable
        self._name = _name

    def apply(self, record: Dict[str, Any]) -> bool:
        """
        Apply filter to a record.

        Args:
            record: Dictionary to filter

        Returns:
            True if record passes filter, False otherwise

        Note:
            Returns False if the field is missing from the record,
            unless checking for None value explicitly.
        """
        # Handle custom callable
        if self._callable is not None:
            return self._callable(record)

        # Get field value, return False if missing (unless checking for None)
        if self.field not in record:
            # Special case: checking if field equals None
            if self.operator == "==" and self.value is None:
                return False
            return False

        field_value = record[self.field]

        # Apply operator
        op_func = self.OPERATORS[self.operator]
        try:
            return op_func(field_value, self.value)
        except (TypeError, ValueError):
            # Handle incompatible comparisons gracefully
            return False

    @classmethod
    def from_callable(cls, fn: Callable[[Dict], bool], name: str = "custom") -> Filter:
        """
        Create filter from arbitrary function.

        Args:
            fn: Function that takes a record dict and returns bool
            name: Name for the filter (for debugging/repr)

        Returns:
            Filter instance wrapping the callable

        Example:
            >>> f = Filter.from_callable(lambda r: r.get("x", 0) > 5)
            >>> f.apply({"x": 10})  # True
        """
        return cls(
            field="",
            operator="custom",
            value=None,
            _callable=fn,
            _name=name,
        )

    @classmethod
    def from_dict(cls, config: Dict[str, Any]) -> Filter:
        """
        Create filter from dictionary configuration.

        Args:
            config: Dict with "field", "op", and "value" keys

        Returns:
            Filter instance

        Raises:
            KeyError: If required keys are missing

        Example:
            >>> f = Filter.from_dict({"field": "month", "op": "<=", "value": 22})
        """
        return cls(
            field=config["field"],
            operator=config["op"],
            value=config["value"],
        )

    def __repr__(self) -> str:
        if self._callable is not None:
            return f"Filter(custom: {self._name or 'anonymous'})"
        return f"Filter({self.field} {self.operator} {self.value!r})"

    def __eq__(self, other: object) -> bool:
        if not isinstance(other, Filter):
            return NotImplemented
        return (
            self.field == other.field
            and self.operator == other.operator
            and self.value == other.value
        )


class FilterChain:
    """
    Combine multiple filters with logical operators.

    FilterChain is immutable - operations return new instances.

    Attributes:
        filters: List of Filter objects
        logic: "AND" (all must pass) or "OR" (any must pass)

    Example:
        >>> chain = FilterChain([
        ...     Filter("month", "<=", 22),
        ...     Filter("dataset", "==", "androzoo")
        ... ], logic="AND")
        >>> chain.apply({"month": 20, "dataset": "androzoo"})  # True
    """

    def __init__(
        self,
        filters: List[Filter],
        logic: Literal["AND", "OR"] = "AND",
    ):
        """
        Create a chain of filters.

        Args:
            filters: List of Filter objects
            logic: "AND" (all must pass) or "OR" (any must pass)

        Example:
            >>> chain = FilterChain([
            ...     Filter("month", "<=", 22),
            ...     Filter("dataset", "==", "androzoo")
            ... ], logic="AND")
        """
        self._filters = list(filters)  # Make a copy for immutability
        self._logic = logic

    @property
    def filters(self) -> List[Filter]:
        """Get copy of filters list."""
        return list(self._filters)

    @property
    def logic(self) -> str:
        """Get logic operator."""
        return self._logic

    def apply(self, record: Dict[str, Any]) -> bool:
        """
        Apply all filters according to logic.

        Args:
            record: Dictionary to filter

        Returns:
            True if record passes filter chain, False otherwise

        Note:
            - Empty AND chain passes everything
            - Empty OR chain fails everything
        """
        if not self._filters:
            # Empty AND passes everything, empty OR fails everything
            return self._logic == "AND"

        if self._logic == "AND":
            return all(f.apply(record) for f in self._filters)
        else:  # OR
            return any(f.apply(record) for f in self._filters)

    def add(self, filter: Filter) -> FilterChain:
        """
        Add a filter to the chain (returns new chain).

        Note: FilterChain is immutable - returns new instance.

        Args:
            filter: Filter to add

        Returns:
            New FilterChain with the filter added
        """
        new_filters = self._filters + [filter]
        return FilterChain(new_filters, self._logic)

    @classmethod
    def from_config(
        cls,
        config: List[Dict],
        logic: Literal["AND", "OR"] = "AND",
    ) -> FilterChain:
        """
        Create chain from list of filter configurations.

        Args:
            config: List of dicts with "field", "op", "value" keys
            logic: "AND" or "OR"

        Returns:
            FilterChain instance

        Example:
            >>> chain = FilterChain.from_config([
            ...     {"field": "month", "op": "<=", "value": 22},
            ...     {"field": "epochs", "op": ">", "value": 10}
            ... ])
        """
        filters = [Filter.from_dict(cfg) for cfg in config]
        return cls(filters, logic)

    def __len__(self) -> int:
        return len(self._filters)

    def __repr__(self) -> str:
        return f"FilterChain({self._logic}, {len(self)} filters)"

    def __iter__(self):
        return iter(self._filters)
