"""
Aurora Framework - Evaluation framework for malware classifiers under drift

Assess classifier stability and reliability across temporal concept drift.
"""

__version__ = "0.1.0"

from .schema import (
    ExperimentResult,
    ExperimentMetadata,
    ResultsCollection,
)

from .ingestion import (
    ResultsLoader,
    PickleResultsLoader,
    JSONResultsLoader,
    ValidationError,
    create_cutoff_month_filter,
    create_hyperparameter_filter,
    combine_collections,
)

from .validation import (
    ResultsValidator,
    ValidationReport,
)

from .metrics import (
    ReliabilityMetrics,
    StabilityMetrics,
    DrawdownMetrics,
    SelectiveClassificationMetrics,
)

from .analyzer import (
    AuroraAnalyzer,
)

from .uncertainty import (
    UncertaintyQualityMetrics,
)

from .data_quality import (
    DataQualityChecker,
    DataQualityReport,
    DataQualityIssue,
    check_results_df,
)

from .base_name_mapping import (
    BaseNameRegistry,
    simple_field_mapper,
    composite_field_mapper,
)

from .ncm import (
    NCMDescriptor,
    NCM_REGISTRY,
    compute_softmax_uncertainty,
    compute_margin_uncertainty,
    expand_result_with_ncms,
    expand_results_with_ncms,
    get_ncms_for_method,
    extract_base_name_from_trainer_mode,
    get_ncm_suffix_from_trainer_mode,
)

from .pareto import (
    ParetoAnalyzer,
    ParetoMetrics,
    ParetoResult,
    UniversalParetoResult,
    pareto_dominates,
    find_pareto_frontier,
    compute_dominance_matrix,
    normalize_metrics,
    compute_pareto_metrics_from_results,
    format_pareto_table,
)

from .rejection import (
    RejectionSimulator,
    RejectionMetrics,
    PostHocRejectorSimulator,
)

from .performance_rejection import (
    PerformanceRejectionSimulator,
    TargetAdherenceMetrics,
    ReliabilityDiagramData,
    MonthlyRejectionResult,
    mann_kendall_test,
    bootstrap_ci,
    run_performance_rejection_analysis,
)

from .tools import (
    compute_metrics_numba,
    compute_aurc,
    RobustPickleIO,
)

# UX Revision Phase 1: Discovery & Filters
from .discovery import (
    FileDiscovery,
)

from .filters import (
    Filter,
    FilterChain,
)

# UX Revision Phase 2: Pipeline
from .pipeline import (
    Pipeline,
)

# UX Revision Phase 3: Config System
from .config import (
    Config,
    ConfigLoader,
    ConfigValidator,
    ConfigValidationError,
    ConfiguredLoader,
    load_with_config,
)

# Core experiment modules available as submodules
# These are optional and may not be available in all environments
# (they require external dependencies like 'uda' and 'augmentations')
try:
    from . import experiments
    _experiments_available = True
except ImportError as e:
    _experiments_available = False
    experiments = None  # type: ignore

try:
    from . import plotting
    _plotting_available = True
except ImportError as e:
    _plotting_available = False
    plotting = None  # type: ignore

try:
    from . import run_parallel
    _run_parallel_available = True
except ImportError as e:
    _run_parallel_available = False
    run_parallel = None  # type: ignore

__all__ = [
    # Schema
    "ExperimentResult",
    "ExperimentMetadata",
    "ResultsCollection",

    # Ingestion
    "ResultsLoader",
    "PickleResultsLoader",
    "JSONResultsLoader",
    "ValidationError",
    "create_cutoff_month_filter",
    "create_hyperparameter_filter",
    "combine_collections",

    # Validation
    "ResultsValidator",
    "ValidationReport",

    # Metrics
    "ReliabilityMetrics",
    "StabilityMetrics",
    "DrawdownMetrics",
    "SelectiveClassificationMetrics",

    # Uncertainty
    "UncertaintyQualityMetrics",

    # Analyzer
    "AuroraAnalyzer",

    # Data Quality
    "DataQualityChecker",
    "DataQualityReport",
    "DataQualityIssue",
    "check_results_df",

    # Base Name Mapping
    "BaseNameRegistry",
    "simple_field_mapper",
    "composite_field_mapper",

    # NCM System
    "NCMDescriptor",
    "NCM_REGISTRY",
    "compute_softmax_uncertainty",
    "compute_margin_uncertainty",
    "expand_result_with_ncms",
    "expand_results_with_ncms",
    "get_ncms_for_method",
    "extract_base_name_from_trainer_mode",
    "get_ncm_suffix_from_trainer_mode",

    # Pareto Analysis
    "ParetoAnalyzer",
    "ParetoMetrics",
    "ParetoResult",
    "UniversalParetoResult",
    "pareto_dominates",
    "find_pareto_frontier",
    "compute_dominance_matrix",
    "normalize_metrics",
    "compute_pareto_metrics_from_results",
    "format_pareto_table",

    # Rejection Simulation (Budget-Based)
    "RejectionSimulator",
    "RejectionMetrics",
    "PostHocRejectorSimulator",

    # Performance-Based Rejection
    "PerformanceRejectionSimulator",
    "TargetAdherenceMetrics",
    "ReliabilityDiagramData",
    "MonthlyRejectionResult",
    "mann_kendall_test",
    "bootstrap_ci",
    "run_performance_rejection_analysis",

    # Core Tools
    "compute_metrics_numba",
    "compute_aurc",
    "RobustPickleIO",

    # UX Revision Phase 1: Discovery & Filters
    "FileDiscovery",
    "Filter",
    "FilterChain",

    # UX Revision Phase 2: Pipeline
    "Pipeline",

    # UX Revision Phase 3: Config System
    "Config",
    "ConfigLoader",
    "ConfigValidator",
    "ConfigValidationError",
    "ConfiguredLoader",
    "load_with_config",

    # Core experiment modules
    "experiments",
    "plotting",
    "run_parallel",
]
